<?php
/** @var $car       \app\models\Car */
/** @var $uploadUrl string */

$this->title = $car->year . ' — ' . $car->info;
$defaultThumb = Yii::$app->request->baseUrl . '/uploads/cars/default.jpg';

$statusMap = [
    'at_auction' => ['label' => 'At Auction', 'class' => 'bg-warning-subtle text-warning border border-warning-subtle'],
    'at_yard'    => ['label' => 'At Yard',    'class' => 'bg-info-subtle text-info border border-info-subtle'],
    'arrived'    => ['label' => 'Arrived',    'class' => 'bg-success-subtle text-success border border-success-subtle'],
    'paid'       => ['label' => 'Paid',       'class' => 'bg-primary-subtle text-primary border border-primary-subtle'],
    'unpaid'     => ['label' => 'Unpaid',     'class' => 'bg-danger-subtle text-danger border border-danger-subtle'],
];
$st = $statusMap[$car->status] ?? ['label' => $car->status, 'class' => 'bg-secondary-subtle text-secondary'];

$imagesByType = ['auction' => [], 'warehouse' => [], 'loading' => []];
foreach ($car->images as $img) {
    if (isset($imagesByType[$img->type])) {
        $imagesByType[$img->type][] = $img;
    }
}

$allImages = $car->images;

// Build JSON array of all image URLs for JS downloader
$allImageUrls = [];
foreach ($allImages as $img) {
    if ($img->image_path) {
        $allImageUrls[] = [
            'url'      => $uploadUrl . $img->image_path,
            'filename' => $img->type . '_' . $img->image_path,
        ];
    }
}

$this->registerCssFile('https://cdn.jsdelivr.net/npm/glightbox/dist/css/glightbox.min.css');
$this->registerJsFile('https://cdn.jsdelivr.net/npm/glightbox/dist/js/glightbox.min.js',   ['position' => \yii\web\View::POS_END]);
$this->registerJsFile('https://cdn.jsdelivr.net/npm/jszip@3.10.1/dist/jszip.min.js',       ['position' => \yii\web\View::POS_END]);
$this->registerJsFile('https://cdn.jsdelivr.net/npm/file-saver@2.0.5/dist/FileSaver.min.js', ['position' => \yii\web\View::POS_END]);
?>

<div class="container-fluid py-3" style="max-width:1100px;">

    <!-- Back -->
    <a href="<?= \yii\helpers\Url::to(['/client/my-cars']) ?>"
       class="btn btn-light rounded-pill px-3 mb-4 shadow-sm">
        <i class="bi bi-arrow-left me-1"></i> Back to My Vehicles
    </a>

    <!-- Title Row -->
    <div class="d-flex justify-content-between align-items-start mb-4 flex-wrap gap-2">
        <div>
            <h4 class="fw-bold mb-2">
                <i class="bi bi-car-front-fill text-primary me-2"></i>
                <?= htmlspecialchars($car->year . ' — ' . $car->info) ?>
            </h4>
            <span class="badge rounded-pill px-3 py-2 <?= $st['class'] ?> me-2">
                <?= $st['label'] ?>
            </span>
            <?php if ($car->shipping_line): ?>
            <a href="<?= htmlspecialchars($car->shipping_line) ?>" target="_blank"
               class="btn btn-sm btn-outline-info rounded-pill">
                <i class="bi bi-truck me-1"></i> Track Shipment
            </a>
            <?php endif; ?>
        </div>
        <div class="text-end">
            <div class="fs-3 fw-bold text-success">$<?= number_format($car->price, 2) ?></div>
            <div class="text-muted small">Purchase Price</div>
        </div>
    </div>

    <div class="row g-4">

        <!-- ── Details ── -->
        <div class="col-lg-5">
            <div class="card border-0 shadow-sm rounded-4">
                <div class="card-header bg-white border-0 px-4 pt-4 pb-0">
                    <h6 class="fw-bold mb-0">
                        <i class="bi bi-card-list text-primary me-2"></i>Vehicle Details
                    </h6>
                </div>
                <div class="card-body px-4 pb-4">
                    <table class="table table-borderless mb-0" style="font-size:0.88rem;">
                        <tbody>
                        <?php
                        $rows = [
                            ['VIN',           htmlspecialchars($car->vin ?: '—')],
                            ['LOT',           '<strong>' . htmlspecialchars($car->lot) . '</strong>'],
                            ['Year',          htmlspecialchars($car->year)],
                            ['Title / Key',   '<span class="badge bg-light text-dark border rounded-pill">' . htmlspecialchars($car->title) . '</span>'],
                            ['Auction',       htmlspecialchars($car->auction    ? $car->auction->name    : '—')],
                            ['Warehouse',     htmlspecialchars($car->warehouse  ? $car->warehouse->name  : '—')],
                            ['Destination',   htmlspecialchars($car->destination ? $car->destination->name : '—')],
                            ['Container',     '<strong>' . htmlspecialchars($car->container ?: '—') . '</strong>'],
                            ['Liner',         htmlspecialchars($car->liner ?: '—')],
                            ['Purchase Date', $car->purchase_date ? date('d/m/Y', strtotime($car->purchase_date)) : '—'],
                            ['Received Date', $car->received_date ? date('d/m/Y', strtotime($car->received_date)) : '—'],
                            ['ETA Date',      '<strong class="text-primary">' . ($car->eta_date ? date('d/m/Y', strtotime($car->eta_date)) : '—') . '</strong>'],
                        ];
                        foreach ($rows as [$label, $value]):
                        ?>
                        <tr>
                            <td class="text-muted ps-0 fw-semibold" style="width:40%"><?= $label ?></td>
                            <td><?= $value ?></td>
                        </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- ── Photos ── -->
        <div class="col-lg-7">

            <!-- Download All Button -->
            <?php if (!empty($allImageUrls)): ?>
            <div class="d-flex justify-content-end mb-3">
                <button id="downloadAllBtn"
                        class="btn btn-success rounded-pill px-4 shadow-sm fw-semibold">
                    <i class="bi bi-download me-2"></i>
                    Download All Photos
                    <span class="badge bg-white text-success ms-1">
                        <?= count($allImageUrls) ?>
                    </span>
                </button>
            </div>

            <!-- Progress bar (hidden by default) -->
            <div id="downloadProgress" class="mb-3 d-none">
                <div class="d-flex justify-content-between small text-muted mb-1">
                    <span>Downloading photos...</span>
                    <span id="downloadProgressText">0%</span>
                </div>
                <div class="progress rounded-pill" style="height:8px;">
                    <div id="downloadProgressBar"
                         class="progress-bar progress-bar-striped progress-bar-animated bg-success"
                         style="width:0%"></div>
                </div>
            </div>
            <?php endif; ?>

            <?php
            $sections = [
                'auction'   => ['label' => 'Auction Photos',   'icon' => 'bi-hammer', 'gallery' => 'gallery-auction'],
                'warehouse' => ['label' => 'Warehouse Photos', 'icon' => 'bi-house',  'gallery' => 'gallery-warehouse'],
                'loading'   => ['label' => 'Loading Photos',   'icon' => 'bi-truck',  'gallery' => 'gallery-loading'],
            ];
            foreach ($sections as $type => $section):
                $imgs = $imagesByType[$type];
            ?>
            <div class="card border-0 shadow-sm rounded-4 mb-3">
                <div class="card-header bg-white border-0 px-4 pt-3 pb-0 d-flex justify-content-between align-items-center">
                    <h6 class="fw-bold mb-3 small text-muted text-uppercase">
                        <i class="bi <?= $section['icon'] ?> me-2"></i>
                        <?= $section['label'] ?>
                        <span class="badge bg-primary-subtle text-primary rounded-pill ms-1">
                            <?= count($imgs) ?>
                        </span>
                    </h6>
                    <?php if (!empty($imgs)): ?>
                    <button class="btn btn-sm btn-outline-primary rounded-pill mb-3 download-section-btn"
                            data-type="<?= $type ?>"
                            data-label="<?= htmlspecialchars($section['label']) ?>">
                        <i class="bi bi-download me-1"></i> Download
                    </button>
                    <?php endif; ?>
                </div>
                <div class="card-body px-4 pb-4 pt-0">
                    <?php if (empty($imgs)): ?>
                    <p class="text-muted small mb-0">
                        <i class="bi bi-image me-1"></i>No photos available.
                    </p>
                    <?php else: ?>
                    <div class="row g-2">
                        <?php foreach ($imgs as $img): ?>
                        <div class="col-6 col-md-4">
                            <a href="<?= $uploadUrl . htmlspecialchars($img->image_path) ?>"
                               class="glightbox"
                               data-gallery="<?= $section['gallery'] ?>"
                               data-description="<?= htmlspecialchars($car->year . ' ' . $car->info . ' — ' . $section['label']) ?>">
                                <img src="<?= $uploadUrl . htmlspecialchars($img->image_path) ?>"
                                     class="img-fluid rounded-3 border shadow-sm w-100 photo-thumb"
                                     style="height:110px; object-fit:cover; cursor:zoom-in; transition:transform 0.2s;"
                                     onerror="this.src='<?= $defaultThumb ?>'">
                            </a>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>

        </div>
    </div>
</div>

<style>
.rounded-4 { border-radius: 1rem !important; }
.table td  { padding: 0.5rem 0.75rem; }
.photo-thumb:hover { transform: scale(1.04); }
</style>

<?php
$carLabel     = htmlspecialchars($car->year . '_' . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $car->info));
$allUrlsJson  = json_encode($allImageUrls, JSON_HEX_TAG | JSON_HEX_QUOT);

// Build per-section image map for section download buttons
$sectionUrlsMap = [];
foreach (['auction', 'warehouse', 'loading'] as $type) {
    $sectionUrlsMap[$type] = [];
    foreach ($imagesByType[$type] as $img) {
        if ($img->image_path) {
            $sectionUrlsMap[$type][] = [
                'url'      => $uploadUrl . $img->image_path,
                'filename' => $img->image_path,
            ];
        }
    }
}
$sectionUrlsJson = json_encode($sectionUrlsMap, JSON_HEX_TAG | JSON_HEX_QUOT);
?>

<?php $this->registerJs(<<<JS
// ── GLightbox ──────────────────────────────────────────────
GLightbox({ touchNavigation: true, loop: true, zoomable: true });

// ── Download helper using JSZip + FileSaver ────────────────
var CAR_LABEL    = '{$carLabel}';
var ALL_IMAGES   = {$allUrlsJson};
var SECTION_IMGS = {$sectionUrlsJson};

async function downloadImages(images, zipName, progressBar, progressText, progressWrap, btn) {
    if (!images || images.length === 0) return;

    btn.disabled = true;
    if (progressWrap) progressWrap.classList.remove('d-none');

    var zip   = new JSZip();
    var total = images.length;
    var done  = 0;

    for (var i = 0; i < images.length; i++) {
        var item = images[i];
        try {
            var response = await fetch(item.url);
            var blob     = await response.blob();
            zip.file(item.filename, blob);
        } catch (e) {
            console.warn('Failed to fetch:', item.url, e);
        }
        done++;
        var pct = Math.round((done / total) * 100);
        if (progressBar)  progressBar.style.width  = pct + '%';
        if (progressText) progressText.textContent  = pct + '%';
    }

    var content = await zip.generateAsync({ type: 'blob' });
    saveAs(content, zipName + '.zip');

    btn.disabled = false;
    if (progressWrap) progressWrap.classList.add('d-none');
    if (progressBar)  progressBar.style.width  = '0%';
    if (progressText) progressText.textContent  = '0%';
}

// ── Download ALL photos button ─────────────────────────────
var downloadAllBtn  = document.getElementById('downloadAllBtn');
var progressWrap    = document.getElementById('downloadProgress');
var progressBar     = document.getElementById('downloadProgressBar');
var progressText    = document.getElementById('downloadProgressText');

if (downloadAllBtn) {
    downloadAllBtn.addEventListener('click', function () {
        downloadImages(ALL_IMAGES, CAR_LABEL + '_all_photos', progressBar, progressText, progressWrap, downloadAllBtn);
    });
}

// ── Download per-section buttons ───────────────────────────
document.querySelectorAll('.download-section-btn').forEach(function (btn) {
    btn.addEventListener('click', function () {
        var type   = btn.getAttribute('data-type');
        var label  = btn.getAttribute('data-label').replace(/\\s+/g, '_');
        var images = SECTION_IMGS[type] || [];
        downloadImages(images, CAR_LABEL + '_' + label, null, null, null, btn);
    });
});
JS, \yii\web\View::POS_END); ?>
